<?php
/*
 * Rose Business Suite - Accounting, CRM and POS Software
 * Copyright (c) UltimateKode.com. All Rights Reserved
 * ***********************************************************************
 *
 *  Email: support@ultimatekode.com
 *  Website: https://www.ultimatekode.com
 *
 *  ************************************************************************
 *  * This software is furnished under a license and may be used and copied
 *  * only  in  accordance  with  the  terms  of such  license and with the
 *  * inclusion of the above copyright notice.
 *  * If you Purchased from Codecanyon, Please read the full License from
 *  * here- http://codecanyon.net/licenses/standard/
 * ***********************************************************************
 */

namespace App\Http\Controllers\Focus\account;

use App\Models\account\Account;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Responses\RedirectResponse;
use App\Http\Responses\ViewResponse;
use App\Http\Responses\Focus\account\CreateResponse;
use App\Http\Responses\Focus\account\EditResponse;
use App\Repositories\Focus\account\AccountRepository;
use App\Http\Requests\Focus\account\ManageAccountRequest;
use App\Http\Requests\Focus\account\StoreAccountRequest;
use App\Models\account\AccountType;
use App\Models\transaction\Transaction;
use Illuminate\Support\Facades\Response;
use Illuminate\Validation\ValidationException;

/**
 * AccountsController
 */
class AccountsController extends Controller
{
    /**
     * variable to store the repository object
     * @var AccountRepository
     */
    protected $repository;

    /**
     * contructor to initialize repository object
     * @param AccountRepository $repository ;
     */
    public function __construct(AccountRepository $repository)
    {
        $this->repository = $repository;
    }

    /**
     * Display a listing of the resource.
     *
     * @param App\Http\Requests\Focus\account\ManageAccountRequest $request
     * @return \App\Http\Responses\ViewResponse
     */
    public function index(ManageAccountRequest $request)
    {
        return new ViewResponse('focus.accounts.index');
    }

    /**
     * Show the form for creating a new resource.
     *
     * @param CreateAccountRequestNamespace $request
     * @return \App\Http\Responses\Focus\account\CreateResponse
     */
    public function create(StoreAccountRequest $request)
    {   
        $account_types = AccountType::get(['id', 'name', 'code']);
        $main_accounts = Account::whereNull('ledger_id')->with('account_type')->get();

        return view('focus.accounts.create', compact('account_types', 'main_accounts'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param StoreAccountRequestNamespace $request
     * @return \App\Http\Responses\RedirectResponse
     */
    public function store(StoreAccountRequest $request)
    {
        try {
            $this->repository->create($request->except('_token'));
        } catch (\Throwable $th) {
            return errorHandler('Error Creating Accounts', $th);
        }

        return new RedirectResponse(route('biller.accounts.index'), ['flash_success' => trans('alerts.backend.accounts.created')]);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param App\Models\account\Account $account
     * @param EditAccountRequestNamespace $request
     * @return \App\Http\Responses\Focus\account\EditResponse
     */
    public function edit(Account $account)
    {
        $account_types = AccountType::get(['id', 'name', 'code']);
        $main_accounts = Account::where('id', '!=', $account->id)->whereNull('ledger_id')->with('account_type')->get();

        return view('focus.accounts.edit', compact('account', 'account_types', 'main_accounts'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param UpdateAccountRequestNamespace $request
     * @param App\Models\account\Account $account
     * @return \App\Http\Responses\RedirectResponse
     */
    public function update(StoreAccountRequest $request, Account $account)
    {
        try {
            $this->repository->update($account, $request->except('_token'));
        } catch (\Throwable $th) {
            return errorHandler('Error Updating Accounts', $th);
        }

        return new RedirectResponse(route('biller.accounts.index'), ['flash_success' => trans('alerts.backend.accounts.updated')]);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param DeleteAccountRequestNamespace $request
     * @param App\Models\account\Account $account
     * @return \App\Http\Responses\RedirectResponse
     */
    public function destroy(Account $account)
    {
        
        try {
            $this->repository->delete($account);
        } catch (\Throwable $th) {
            return errorHandler('Error Deleting Account', $th);
        }

        return new RedirectResponse(route('biller.accounts.index'), ['flash_success' => trans('alerts.backend.accounts.deleted')]);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param DeleteAccountRequestNamespace $request
     * @param App\Models\account\Account $account
     * @return \App\Http\Responses\RedirectResponse
     */
    public function show(Account $account)
    {
        return view('focus.accounts.view', compact('account'));
    }

    /**
     * Search next account number
     */
    public function search_next_account_no(Request $request)
    {
        $account_type = $request->account_type;
        $number = Account::where('account_type', $account_type)->max('number');

        $series = accounts_numbering($account_type);
        if ($number) $series = $number + 1;
            
        return response()->json(['account_number' => $series]);
    }    

    /**
     * Search Expense accounts 
     */
    public function account_search(Request $request)
    {
        if (!access()->allow('product_search')) return false;

        $k = $request->keyword;
        
        if ($request->type == 'Expense') {
            $accounts = Account::where('account_type', 'Expense')
            ->where(function ($q) use($k) {
                $q->where('name', 'LIKE', '%' . $k . '%')->orWhere('number', 'LIKE', '%' . $k . '%');
            })->limit(6)->get(['id', 'holder AS name', 'number']);

            return response()->json($accounts);
        }

        $accounts = Account::where('name', 'LIKE', '%' . $k . '%')
            ->orWhere('number', 'LIKE', '%' . $k . '%')
            ->limit(6)->get(['id', 'holder AS name', 'number']);

        return response()->json($accounts);
    }

    /**
     * Profit and Loss (Income)
     */
    public function profit_and_loss(Request $request)
    {
        $dates = $request->only('start_date', 'end_date');
        $dates = array_map(fn($v) =>  date_for_database($v), $dates);

        $q = Account::whereHas('transactions', function ($q) use($dates) {
                $q->when($dates, function ($q) use($dates) {
                    $q->whereBetween('tr_date', $dates);
                });
            });

        $accounts = $q->get();

        if ($request->type == 'p') {
            return $this->print_document('profit_and_loss', $accounts, $dates, 0);
        } 

        $bg_styles = [
            'bg-gradient-x-info', 'bg-gradient-x-purple', 'bg-gradient-x-grey-blue', 'bg-gradient-x-danger',
        ];
            
        return new ViewResponse('focus.accounts.profit_&_loss', compact('accounts', 'bg_styles', 'dates'));
    }

    /**
     * Balance Sheet
     */
    public function balance_sheet(Request $request)
    {
        $date = date_for_database(request('end_date'));

        $bal_sheet_q = Account::query();
        $profit_loss_q = Account::query();
        if (request('end_date')) {
            // balance sheet accounts
            $bal_sheet_q->whereIn('ledger_type', ['asset', 'equity', 'liability'])
                ->whereHas('account_type', fn($q) => $q->whereNotIn('code', ['income', 'cog', 'operating_expense']))
                ->whereHas('transactions', function ($q) use($date) {
                    $q->whereDate('tr_date', '<=', $date);
                });
            // profit & loss accounts
            $profit_loss_q
                ->whereHas('account_type', fn($q) => $q->whereIn('code', ['income', 'cog', 'operating_expense']))
                ->whereHas('transactions', function ($q) use($date) {
                    $q->where('tr_date', '<=', $date);
                });
        } else {
            // balance sheet accounts
            $bal_sheet_q->whereHas('transactions')->whereIn('ledger_type', ['asset', 'equity', 'liability'])
                ->whereHas('account_type', fn($q) => $q->whereNotIn('code', ['income', 'cog', 'operating_expense']));
            // profit & loss accounts
            $profit_loss_q->whereHas('transactions')
            ->whereHas('account_type', fn($q) => $q->whereIn('code', ['income', 'cog', 'operating_expense']));
        }

        // compute profit and loss
        $net_profit = 0;
        $net_accounts = $profit_loss_q->get();
        foreach ($net_accounts as $account) {
            $debit = $account->transactions()->sum('debit');
            $credit = $account->transactions()->sum('credit');
            $ledger_categ = @$account->account_type->code;
            if ($ledger_categ == 'income') {
                $credit_balance = round($credit - $debit, 2);
                $net_profit += $credit_balance;
            } elseif (in_array($ledger_categ, ['cog', 'operating_expense'])) {
                $debit_balance = round($debit - $credit, 2);
                $net_profit -= $debit_balance;
            }
        }

        // fetch balance sheet accounts
        $accounts = $bal_sheet_q->get();
        $bg_styles = ['bg-gradient-x-info', 'bg-gradient-x-purple', 'bg-gradient-x-grey-blue', 'bg-gradient-x-danger'];

        // print balance_sheet
        if ($request->type == 'p') return $this->print_document('balance_sheet', $accounts, array(0, $date), $net_profit);       
            
        return new ViewResponse('focus.accounts.balance_sheet', compact('accounts', 'bg_styles', 'net_profit', 'date'));
    }

    /**
     * Trial Balance
     */
    public function trial_balance(Request $request)
    {   
        $end_date = $request->end_date? date_for_database($request->end_date) : '';
        $q = Account::whereHas('transactions', function ($q) use($end_date) {
            $q->when($end_date, function ($q) use($end_date) {
                $q->whereDate('tr_date', '<=', $end_date);
            });
        });
    
        $accounts = $q->orderBy('number', 'asc')->get();
        $date = date_for_database($end_date);
        if ($request->type == 'p') 
            return $this->print_document('trial_balance', $accounts, [0, $date], 0);
        
        return new ViewResponse('focus.accounts.trial_balance', compact('accounts', 'date'));
    }

    /**
     * Print document
     */
    public function print_document(string $name, $accounts, array $dates, float $net_profit)
    {
        $account_types = ['Assets', 'Equity', 'Expenses', 'Liabilities', 'Income'];
        $params = compact('accounts', 'account_types', 'dates', 'net_profit');
        $html = view('focus.accounts.print_' . $name, $params)->render();
        $pdf = new \Mpdf\Mpdf(config('pdf'));
        $pdf->WriteHTML($html);
        $headers = array(
            "Content-type" => "application/pdf",
            "Pragma" => "no-cache",
            "Cache-Control" => "must-revalidate, post-check=0, pre-check=0",
            "Expires" => "0"
        );
        return Response::stream($pdf->Output($name . '.pdf', 'I'), 200, $headers);
    }

    /**
     * Cashbook Index
     * 
     */
    public function cashbook()
    {
        $accounts = Account::whereHas('account_type', fn($q) => $q->where('code', 'bank'))->get(['id', 'name']);

        return new ViewResponse('focus.accounts.cashbook', compact('accounts'));
    }

    // cashbook transactions
    static function cashbook_transactions()
    {
        $q = Transaction::whereHas('account', fn($q) => $q->whereHas('account_type', fn($q) => $q->where('code', 'bank')))
        ->where(function ($q) {
            $q->whereHas('payment')
            ->orWhereHas('bank_charge')
            ->orWhereHas('bank_transfer');
        })
        ->when(request('ledger_id'), fn($q) => $q->where('ledger_id', request('ledger_id')))
        ->when(request('tr_type') == 'receipt', fn($q) => $q->where('debit', '>', 0))
        ->when(request('tr_type') == 'payment', fn($q) => $q->where('credit', '>', 0))
        ->when(request('start_date') && request('end_date'), function ($q) {
            $q->whereBetween('date', [
                date_for_database(request('start_date')),
                date_for_database(request('end_date')),
            ]);
        });

        return $q;
    }
}
