<?php
/*
 * Rose Business Suite - Accounting, CRM and POS Software
 * Copyright (c) UltimateKode.com. All Rights Reserved
 * ***********************************************************************
 *
 *  Email: support@ultimatekode.com
 *  Website: https://www.ultimatekode.com
 *
 *  ************************************************************************
 *  * This software is furnished under a license and may be used and copied
 *  * only  in  accordance  with  the  terms  of such  license and with the
 *  * inclusion of the above copyright notice.
 *  * If you Purchased from Codecanyon, Please read the full License from
 *  * here- http://codecanyon.net/licenses/standard/
 * ***********************************************************************
 */

namespace App\Http\Controllers\Focus\invoice;

use App\Http\Controllers\Focus\printer\RegistersController;
use App\Http\Requests\Focus\invoice\ManagePosRequest;
use App\Models\account\Account;
use App\Models\Company\ConfigMeta;
use App\Models\customer\Customer;
use App\Models\invoice\Invoice;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Responses\ViewResponse;
use App\Http\Responses\Focus\invoice\CreateResponse;
use App\Http\Responses\Focus\invoice\EditResponse;
use App\Repositories\Focus\invoice\InvoiceRepository;
use App\Http\Requests\Focus\invoice\ManageInvoiceRequest;
use App\Http\Requests\Focus\invoice\CreateInvoiceRequest;
use App\Http\Requests\Focus\invoice\EditInvoiceRequest;
use App\Http\Responses\RedirectResponse;
use App\Models\additional\Additional;
use App\Models\Company\Company;
use App\Models\currency\Currency;
use App\Models\term\Term;
use App\Repositories\Focus\pos\PosRepository;
use Endroid\QrCode\QrCode;
use Error;
use Illuminate\Validation\ValidationException;
use Storage;

/**
 * InvoicesController
 */
class InvoicesController extends Controller
{
    /**
     * variable to store the repository object
     * @var InvoiceRepository
     */
    protected $repository;
    protected $pos_repository;

    /**
     * contructor to initialize repository object
     * @param InvoiceRepository $repository ;
     */
    public function __construct(
        InvoiceRepository $repository,
        PosRepository $pos_repository
    )
    {
        $this->repository = $repository;
        $this->pos_repository = $pos_repository;
    }

    /**
     * Display a listing of the resource.
     *
     * @param App\Http\Requests\Focus\invoice\ManageInvoiceRequest $request
     * @return \App\Http\Responses\ViewResponse
     */
    public function index(ManageInvoiceRequest $request)
    {
        $customers = Customer::whereHas('invoices')->get(['id', 'company']);

        return new ViewResponse('focus.invoices.index', compact('customers'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @param CreateInvoiceRequestNamespace $request
     * @return \App\Http\Responses\Focus\invoice\CreateResponse
     */
    public function create(CreateInvoiceRequest $request)
    {
        return new CreateResponse('focus.invoices.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param StoreInvoiceRequestNamespace $request
     * @return \App\Http\Responses\RedirectResponse
     */
    public function store(CreateInvoiceRequest $request)
    {
        dd($request->all());
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param App\Models\invoice\Invoice $invoice
     * @param EditInvoiceRequestNamespace $request
     * @return \App\Http\Responses\Focus\invoice\EditResponse
     */
    public function edit(Invoice $invoice, EditInvoiceRequest $request)
    {
        return new EditResponse($invoice);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param DeleteInvoiceRequestNamespace $request
     * @param App\Models\invoice\Invoice $invoice
     * @return \App\Http\Responses\RedirectResponse
     */
    public function destroy(Invoice $invoice)
    {
        try {
            $this->repository->delete($invoice);
        } catch (\Throwable $th) {
            return errorHandler('Error Deleting Invoice', $th);
        }

        return new RedirectResponse(route('biller.invoices.index'), ['flash_success' => trans('alerts.backend.invoices.deleted')]);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param DeleteInvoiceRequestNamespace $request
     * @param App\Models\invoice\Invoice $invoice
     * @return \App\Http\Responses\RedirectResponse
     */
    public function show(Invoice $invoice, ManageInvoiceRequest $request)
    {
        $accounts = Account::all();
        $features = ConfigMeta::where('feature_id', 9)->first();
        $invoice['bill_type'] = 1;
        $words = [
            'prefix' => '',
            'paynote' => trans('invoices.payment_for_invoice') . ' '. '#' . $invoice->tid
        ];

        return new ViewResponse('focus.invoices.view', compact('invoice', 'accounts', 'features', 'words'));
    }

    /**
     * Work Order Invoice
     * 
     */
    function raise_invoice(Request $request) 
    {
        if ($request['_token'] != csrf_token()) abort(401, 'Unauthrozied');

        try {
            $invoice = $this->repository->create($request->except('_token'));
            // print preview url
            $valid_token = token_validator('', 'q'.$invoice->id .$invoice->tid, true);
            $url = ' <a href="'. route('biller.print_bill', [$invoice->id, 4, $valid_token, 1]) .'" class="invisible" id="printpreview"></a>';
        } catch (\Throwable $th) {
            dd($th);
            return errorHandler('Error creating Invoice', $th);
        }

        return new RedirectResponse(route('biller.invoices.index'), ['flash_success' => 'Work Order invoice created successfully' . $url]);
    }


    /**
     * Select drop down invoices
     */
    public function select(Request $request)
    {
        $kw = $request->search;
        if ($kw) {
            $invoices = Invoice::when(request('customer_id'), fn($q) => $q->where('customer_id', request('customer_id')))
            ->when(request('supplier_id'), fn($q) => $q->where('supplier_id', request('supplier_id')))
            ->where(fn($q) =>  $q->where('tid', 'LIKE', "%{$kw}%")->orWhere('note', 'LIKE', "%{$kw}%"))
            ->orderBy('due_date', 'asc')->limit(6)
            ->get();
        } else {
            $invoices = Invoice::when(request('customer_id'), fn($q) => $q->where('customer_id', request('customer_id')))
            ->when(request('supplier_id'), fn($q) => $q->where('supplier_id', request('supplier_id')))
            ->when(request('currency_id'), fn($q) => $q->where('currency_id', request('currency_id')))
            ->when(request('start_date') && request('end_date'), function($q) {
                $q->whereBetween('due_date', [
                    date_for_database(request('start_date')), 
                    date_for_database(request('end_date'))
                ]);
            })
            ->orderBy('due_date', 'asc')
            ->get()
            ->map(function($v) {
                $v['amount_received'] = $v->amount_received;
                $v['amount_due'] = $v->total - ($v->amount_paid + $v['amount_received']);
                return $v;
            })
            ->filter(fn($v) => $v->amount_due > 0);
        }

        return response()->json($invoices);
    }


    /**
     * Print invoice payment receipt
     */
    public function print_payment(InvoicePayment $paidinvoice)
    {
        $html = view('focus.invoices.print_payment', ['resource' => $paidinvoice])->render();
        $pdf = new \Mpdf\Mpdf(config('pdf'));
        $pdf->WriteHTML($html);
        $headers = array(
            "Content-type" => "application/pdf",
            "Pragma" => "no-cache",
            "Cache-Control" => "must-revalidate, post-check=0, pre-check=0",
            "Expires" => "0"
        );

        return Response::stream($pdf->Output('payment.pdf', 'I'), 200, $headers);
    }

    /**
     * POS Create
     */
    public function pos(ManagePosRequest $request, RegistersController $register)
    {
        if (!$register->status()) return view('focus.invoices.pos.open_register');

        $tid = Invoice::where('ins', auth()->user()->ins)->max('tid');
        $customer = Customer::first();
        $currencies = Currency::all();
        $terms = Term::all();
        $additionals = Additional::all();
        $defaults = ConfigMeta::get()->groupBy('feature_id');
        
        $pos_account = Account::where('system', 'pos')->first(['id', 'name']);
        $accounts = Account::where('account_type', 'Asset')
            ->whereHas('account_type', fn($q) => $q->where('system', 'bank'))
            ->get(['id', 'name', 'number']);
        
        $params = compact('customer', 'accounts', 'pos_account', 'tid', 'currencies', 'terms', 'additionals', 'defaults');
        return view('focus.invoices.pos.create', $params)->with(product_helper());
    }

    /**
     * POS Store
     */
    public function pos_store(CreateInvoiceRequest $request)
    {
        if (request('is_pay') && (!request('pmt_reference') || !request('p_account'))) {
            throw ValidationException::withMessages(['payment reference and payment account is required!']);
        }

        // dd($request->all());
        try {
            $result = $this->pos_repository->create($request->except('_token'));
        } catch (\Throwable $th) {
            return errorHandler('Error Creating POS Transaction', $th);
        }

        return response()->json([
            'status' => 'Success',
            'message' => 'POS Transaction Done Successfully',
            'invoice' => $result,
            // 'invoice' => (object) ['id' => 62],
        ]);
    }

    /**
     * TIMS KIT Api Call: Electronic Tax Register (ETR) Invoice
     */
    public function attach_etr()
    {
        $company = Company::find(auth()->user()->ins);
        if (!$company->etr_invoice_endpoint) throw new Error('ETR invoice endpoint not set!');

        $invoice = Invoice::find(request('invoice_id'));
        if (!array_key_exists('etr_url', $invoice->toArray())) throw new Error('ETR invoice url field not set!');
        if (!array_key_exists('etr_qrcode', $invoice->toArray())) throw new Error('ETR invoice QRcode field not set!');

        $payload = config('datecs_etr.invoice');

        if ($invoice->customer) {
            $customer = $invoice->customer;
            $payload['buyer'] = [
                'buyerAddress' => 'string',
                'buyerName' => $customer->company,
                'buyerPhone' => $customer->phone,
                'pinOfBuyer' => 'P123456789P',
            ];
        }
        $payload['items'][0]['unitPrice'] = +$invoice->total;
        $payload['payment'][0]['amount'] = +$invoice->total;

        try {
            $client = new \GuzzleHttp\Client();
            $client_resp = $client->post($company->etr_invoice_endpoint, [
                'headers' => [
                    'Content-Type' => "application/json",
                    'Accept' => "application/json",
                ],
                'json' => $payload,
            ]);
            $data = json_decode($client_resp->getBody()->getContents());

            if ($data->messages == 'Success' && isset($data->verificationUrl)) {
                // extract invoice no
                $query = parse_url($data->verificationUrl, PHP_URL_QUERY);
                parse_str($query, $params);
                $invoice_no = $params['invoiceNo'];
                // generate QR code
                $timestamp = date('Y_m_d_H_i_s');
                $filename = "invoice_{$invoice_no}_{$timestamp}.png";
                $qrCode = new QrCode($data->verificationUrl);
                $qrCode->writeFile(Storage::disk('public')->path("qr".DIRECTORY_SEPARATOR."{$filename}"));
                // update invoice
                $invoice->update(['etr_url' => $data->verificationUrl, 'etr_qrcode' => $filename]);
            }

            return (array) $data;
        } catch (\Throwable $th) {
            printlog($th->getMessage());
            throw new Error('ETR Processing error!');
        }
    }
}
