<?php
/*
 * Rose Business Suite - Accounting, CRM and POS Software
 * Copyright (c) UltimateKode.com. All Rights Reserved
 * ***********************************************************************
 *
 *  Email: support@ultimatekode.com
 *  Website: https://www.ultimatekode.com
 *
 *  ************************************************************************
 *  * This software is furnished under a license and may be used and copied
 *  * only  in  accordance  with  the  terms  of such  license and with the
 *  * inclusion of the above copyright notice.
 *  * If you Purchased from Codecanyon, Please read the full License from
 *  * here- http://codecanyon.net/licenses/standard/
 * ***********************************************************************
 */
namespace App\Http\Controllers\Focus\supplier;

use App\Http\Controllers\Controller;
use Yajra\DataTables\Facades\DataTables;
use App\Repositories\Focus\supplier\SupplierRepository;
use App\Http\Requests\Focus\supplier\ManageSupplierRequest;

/**
 * Class SuppliersTableController.
 */
class SuppliersTableController extends Controller
{
    /**
     * variable to store the repository object
     * @var SupplierRepository
     */
    protected $supplier;
    protected $balance = 0;

    /**
     * contructor to initialize repository object
     * @param SupplierRepository $supplier ;
     */
    public function __construct(SupplierRepository $supplier)
    {
        $this->supplier = $supplier;
    }

    /**
     * This method return the data of the model
     * @param ManageSupplierRequest $request
     *
     * @return mixed
     */
    public function __invoke(ManageSupplierRequest $request)
    {
        if (request('is_transaction')) return $this->invoke_transaction();
        if (request('is_bill')) return $this->invoke_bill();
        if (request('is_statement')) return $this->invoke_statement();
            
        $core = $this->supplier->getForDataTable();
        return Datatables::of($core)
            ->escapeColumns(['id'])
            ->addIndexColumn()
            ->addColumn('name', function ($supplier) {
                return '<a class="font-weight-bold" href="' . route('biller.suppliers.show', $supplier) . '">' . $supplier->name . '</a>';
            })
            ->make(true);
    }

    /**
     * Statement on Account
     */
    public function invoke_transaction()
    {
        $core = $this->supplier->getTransactionsForDataTable();

        return Datatables::of($core)
        ->escapeColumns(['id'])
        ->addIndexColumn()
        ->addColumn('date', function ($tr) {
            return '<span sortid="'. strtotime($tr->date) .'">'. dateFormat($tr->date) .'</span>';
        })
        ->addColumn('type', function ($tr) {
            return $tr->origin_type;
        })
        ->addColumn('note', function ($tr) {
            $note = '';
            switch ($tr->origin_type) {
                case 'manual_journal': $note = "({$tr->manual_journal->tid_code}) {$tr->manual_journal->note}"; break;
                case 'bill': $note = "({$tr->bill->tid_code}) {$tr->bill->note}"; break;
                case 'payment': $note = "({$tr->payment->tid_code}) {$tr->payment->note}"; break;
            }
            return $note;
        })
        ->addColumn('bill_amount', function ($tr) {
            $credit = $tr->frx_rate > 1? $tr->frx_credit : $tr->credit;
            return numberFormat($credit);
        })
        ->addColumn('amount_paid', function ($tr) {
            $debit = $tr->frx_rate > 1? $tr->frx_debit : $tr->debit;
            return numberFormat($debit);
        })
        ->addColumn('account_balance', function ($tr) {
            $debit = $tr->frx_rate > 1? $tr->frx_debit : $tr->debit;
            $credit = $tr->frx_rate > 1? $tr->frx_credit : $tr->credit;
            if ($debit > 0) $this->balance -= $debit;
            elseif ($credit > 0) $this->balance += $credit;

            return numberFormat($this->balance);
        })
        ->make(true);
    }

    /**
     * Bills report
     *  */ 
    public function invoke_bill()
    {
        $core = $this->supplier->getBillsForDataTable();
        
        return Datatables::of($core)
        ->escapeColumns(['id'])
        ->addIndexColumn()
        ->addColumn('date', function ($bill) {
            return dateFormat($bill->date);
        })
        ->addColumn('status', function ($bill) {
            return $bill->status;
        })
        ->addColumn('note', function ($bill) {
            return "({$bill->tid_code}) {$bill->note}";
        })
        ->addColumn('amount', function ($bill) {
            return numberFormat($bill->total);
        })
        ->addColumn('paid', function ($bill) {
            return numberFormat($bill->amount_received);
        })
        ->make(true);
    }

    /**
     * Statement on Invoice
     */
    public function invoke_statement()
    {
        $core = $this->supplier->getStatementForDataTable();
        
        return Datatables::of($core)
        ->escapeColumns(['id'])
        ->addIndexColumn()
        ->addColumn('date', function ($statement) {
            return dateFormat($statement->date);
        })
        ->addColumn('type', function ($statement) {
            $type = $statement->type;
            if ($type == 'bill') $type = '<a href="'. route('biller.invoices.show', $statement->bill_id) .'">'. $type .'</a>';

            return $type;
        })
        ->addColumn('note', function ($statement) {
            return $statement->note;
        })
        ->addColumn('bill_amount', function ($statement) {
            return numberFormat($statement->credit);
        })
        ->addColumn('amount_paid', function ($statement) {
            return numberFormat($statement->debit);
        })
        ->addColumn('bill_balance', function ($statement) {
            if ($statement->type == 'bill') 
                $this->balance = $statement->credit;
            else $this->balance -= $statement->debit;
            return numberFormat($this->balance);
        })
        ->make(true);
    }
}
