<?php

namespace App\Http\Controllers\Focus\work_order;

use App\Http\Controllers\Controller;
use App\Http\Requests\Focus\work_order\CreateWorkOrderRequest;
use App\Http\Requests\Focus\work_order\ManageWorkOrderRequest;
use App\Repositories\Focus\work_order\WorkOrderRepository;
use App\Http\Responses\Focus\work_order\CreateResponse;
use App\Http\Responses\Focus\work_order\EditResponse;
use App\Http\Responses\ViewResponse;
use App\Http\Responses\RedirectResponse;
use App\Models\account\Account;
use App\Models\additional\Additional;
use App\Models\bank\Bank;
use App\Models\work_order\WorkOrder;
use App\Models\Company\Company;
use App\Models\currency\Currency;
use App\Models\customer\Customer;
use App\Models\term\Term;
use Carbon\Carbon;
use Illuminate\Http\Request;
use DB;
use Illuminate\Validation\ValidationException;

class WorkOrderController extends Controller
{
    /**
     * variable to store the repository object
     * @var WorkOrderRepository
     */
    protected $repository;

    /**
     * contructor to initialize repository object
     * @param WorkOrderRepository $repository ;
     */
    public function __construct(WorkOrderRepository $repository)
    {
        $this->repository = $repository;
    }

    /**
     * Display a listing of the resource.
     *
     * @param App\Http\Requests\Focus\work_order\ManageBankRequest $request
     * @return \App\Http\Responses\ViewResponse
     */
    public function index(Request $request)
    {
        return new ViewResponse('focus.work_orders.index');
    }

    /**
     * Approved work orders
     */
    public function approved_work_order(Request $request)
    {
        return new ViewResponse('focus.work_orders.approved_work_order');
    }

    /**
     * Show the form for creating a new resource.
     *
     * @param CreateBankRequestNamespace $request
     * @return \App\Http\Responses\Focus\work_order\CreateResponse
     */
    public function create(Request $request)
    {
        $tid = (new WorkOrder())->next_tid_code;

        $customers = Customer::where('id', '>', 1)->get(['id', 'company']);
        $banks = Bank::all();
        $accounts = Account::whereHas('account_type', fn($q) => $q->whereIn('name', ['Income', 'Other Income']))->get();
        $terms = Term::where('type', 1)->get();  // invoice term type is 1
        $tax_rates = Additional::all();
        $currencies = Currency::all();
        
        return new ViewResponse('focus.work_orders.create', compact('tid', 'customers', 'banks', 'accounts', 'terms', 'tax_rates', 'currencies'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param StoreBankRequestNamespace $request
     * @return \App\Http\Responses\RedirectResponse
     */
    public function store(Request $request)
    {
        try {
            $this->repository->create($request->except('_token'));
        } catch (\Throwable $th) { 
            dd($th);
            return errorHandler('Error Creating Invoice', $th);
        }

        return new RedirectResponse(route('biller.work_orders.index'), ['flash_success' => 'Work Order successfully created']);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param App\Models\work_order\WorkOrder $work_order
     * @return \App\Http\Responses\Focus\work_order\EditResponse
     */
    public function edit(WorkOrder $workOrder, Request $request)
    {
        return new ViewResponse('focus.work_orders.edit', compact('workOrder'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param App\Models\work_order\WorkOrder $work_order
     * @return \App\Http\Responses\RedirectResponse
     */
    public function update(Request $request, WorkOrder $work_order)
    {
        dd($request->all());
        $request->validate([
            'name' => 'required|string',
            'bank' => 'required|string',
            'number' => 'required'
        ]);
        //Input received from the request
        $input = $request->except(['_token', 'ins']);
        try {
            //Update the model using repository update method
            $this->repository->update($work_order, $input);
        } catch (\Throwable $th) {
            return errorHandler('Error Updating Standard Invoice', $th);
        }
        //return with successfull message
        return new RedirectResponse(route('biller.work_orders.index'), ['flash_success' => 'Work Order successfully updated']);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param App\Models\work_order\WorkOrder $work_order
     * @return \App\Http\Responses\RedirectResponse
     */
    public function destroy(WorkOrder $work_order, Request  $request)
    {
        try {
            $this->repository->delete($work_order);
        } catch (\Throwable $th) {
            return errorHandler('Error Deleting Standard Invoice', $th);
        }
        return new RedirectResponse(route('biller.work_orders.index'), ['flash_success' => 'Work Order successfully deleted']);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param DeleteBankRequestNamespace $request
     * @param App\Models\work_order\WorkOrder $work_order
     * @return \App\Http\Responses\RedirectResponse
     */
    public function show(WorkOrder $work_order, Request $request)
    {
        return new ViewResponse('focus.work_orders.view', compact('work_order'));
    }

    /**
     * Create Customer
     */
    public function create_customer(Request $request)
    {
        // dd(request()->all());
        $request->validate([
            'company' => 'required',
        ]);
        $input = $request->only(['company', 'name', 'email', 'phone', 'address', 'tax_pin']);

        $is_company = Customer::where('company', $input['company'])->count();
        if ($is_company) throw ValidationException::withMessages(['Company already exists!']);
        $email_exists = Customer::where('email', $input['email'])->whereNotNull('email')->count();
        if ($email_exists) throw ValidationException::withMessages(['Email already exists!']);
        $taxid_exists = Customer::where('taxid', $input['tax_pin'])->whereNotNull('taxid')->count();
        if ($taxid_exists) throw ValidationException::withMessages(['Tax Pin already exists!']);
        $is_company = Company::where(['id' => auth()->user()->ins, 'taxid' => $input['tax_pin']])->whereNotNull('taxid')->count();
        if ($is_company) throw ValidationException::withMessages(['Company Tax Pin is not allowed!']);
        
        try {
            $input['taxid'] = $input['tax_pin'];
            unset($input['tax_pin']);
            Customer::create($input);
        } catch (\Throwable $th) {
            throw ValidationException::withMessages(['Error creating customer']);
        }

        return redirect()->back()->with('flash_success', 'Customer Created Successfully');
    }

    /**
     * Status Update
     */
    function status_update(Request $request) 
    {
        try {
            $work_order = WorkOrder::findOrFail($request->work_order_id);
            $work_order->update([
                'status' => $request->status,
                'status_note' => $request->status_note,
                'status_mod_by' => auth()->user()->id,
            ]);
        } catch (\Throwable $th) {
            return redirect()->back()->with(['flash_error' => 'Error updating status']);
        }

        return redirect()->back()->with(['flash_success' => 'Status updated successfully']);
    }    
}
