<?php

namespace App\Repositories\Focus\invoice_payment;

use App\Exceptions\GeneralException;
use App\Models\invoice_payment\InvoicePayment;
use App\Models\items\InvoicePaymentItem;
use App\Repositories\BaseRepository;
use App\Repositories\Focus\JournalEntryService;
use DB;
use Illuminate\Support\Arr;
use Illuminate\Validation\ValidationException;

class InvoicePaymentRepository extends BaseRepository
{
    use JournalEntryService;
    /**
     * Associated Repository Model.
     */
    const MODEL = InvoicePayment::class;

    /**
     * This method is used by Table Controller
     * For getting the table data to show in
     * the grid
     * @return mixed
     */
    public function getForDataTable()
    {
        $q = $this->query();

        $q->when(request('customer_id'), function ($q) {
            $q->where('customer_id', request('customer_id'));
        });
            
        return $q;
    }

    /**
     * For Creating the respective model in storage
     *
     * @param array $input
     * @throws GeneralException
     * @return InvoicePayment $payment
     */
    public function create(array $input)
    {
        // dd($input);
        DB::beginTransaction();

        foreach ($input as $key => $val) {
            if ($key == 'date') $input[$key] = date_for_database($val);
            if (in_array($key, ['frx_rate', 'amount', 'allocated_total', 'unallocated_total', 'row_amount'])) {
                if (is_array($val)) $input[$key] = array_map(fn($v) => numberClean($v), $val);
                else $input[$key] = numberClean($val);
            }
        }

        $data_items = modify_array(Arr::only($input, ['inv_id', 'row_amount']));
        if (!$data_items && $input['amount'] == 0) 
            throw ValidationException::withMessages(['amount is required']);
        // over payment line item 
        if ($input['unallocated_total'] > 0) {
            if (round($input['amount']) > round($input['unallocated_total'])) {
                $data_items[] = ['row_amount' => $input['unallocated_total'], 'inv_id' => null];
            }
        }
        
        // payment data
        $invoice_payment = InvoicePayment::create($input);

        // payment data items
        $frx_rate = $invoice_payment->frx_rate;
        $data_items = array_map(fn($v) => [
            'payment_id' => $invoice_payment->id,
            'invoice_id' => $v['inv_id'],
            'rate' => $v['row_amount'],
            'amount' => $v['row_amount'],
            'frx_rate' => $frx_rate > 1? $frx_rate * $v['row_amount'] : 0,  
            'frx_amount' => $frx_rate > 1? $frx_rate * $v['row_amount'] : 0,
            'user_id' => $invoice_payment->user_id,
            'ins' => $invoice_payment->ins,
        ], $data_items);
        InvoicePaymentItem::insert($data_items);
        
        // compute gain/loss on foreign exchange
        if ($frx_rate > 1) {
            $invoice_payment->update(['frx_amount' => $invoice_payment->items()->sum('frx_amount')]);
            $invoice_amount = 0;
            foreach ($invoice_payment->items as $item) {
                if ($item->invoice) {
                    $invoice_amount += ($item->invoice->frx_rate * $item['frx_rate']);
                }
            }
            $invoice_payment['frx_gain'] = $invoice_payment->frx_amount - $invoice_amount;
            $invoice_payment['invoice_amount'] = $invoice_amount;
        }

        // invoice payment accounting
        $invoice_payment['customer_deposit'] = true;
        $invoice_payment['pre_payment'] = !$data_items;
        $this->invoice_payment_acc($invoice_payment, 'store');

        if ($invoice_payment) {
            DB::commit();
            return $invoice_payment;
        }
    }

    /**
     * For updating the respective Model in storage
     *
     * @param InvoicePayment $invoice_payment
     * @param array $input
     * @throws GeneralException
     * return bool
     */
    public function update($invoice_payment, array $input)
    {
        // dd($input);
        DB::beginTransaction();

        foreach ($input as $key => $val) {
            if ($key == 'date') $input[$key] = date_for_database($val);
            if (in_array($key, ['frx_rate', 'amount', 'allocated_total', 'unallocated_total', 'row_amount'])) {
                if (is_array($val)) $input[$key] = array_map(fn($v) => numberClean($v), $val);
                else $input[$key] = numberClean($val);
            }
        }

        $data_items = modify_array(Arr::only($input, ['id', 'inv_id', 'row_amount']));
        if (!$data_items && $input['amount'] == 0) 
            throw ValidationException::withMessages(['amount is required']);
        // over payment line item 
        if ($input['unallocated_total'] > 0) {
            if (round($input['amount']) > round($input['unallocated_total'])) {
                $data_items[] = ['row_amount' => $input['unallocated_total'], 'inv_id' => null, 'id' => null];
            }
        }
        
        // payment data
        $invoice_payment->update($input);

        // payment data items
        $frx_rate = $invoice_payment->frx_rate;
        $data_items = array_map(fn($v) => [
            'id' => $v['id'],
            'payment_id' => $invoice_payment->id,
            'invoice_id' => $v['inv_id'],
            'rate' => $v['row_amount'],
            'amount' => $v['row_amount'],
            'frx_rate' => $frx_rate > 1? $frx_rate * $v['row_amount'] : 0,  
            'frx_amount' => $frx_rate > 1? $frx_rate * $v['row_amount'] : 0,
            'user_id' => $invoice_payment->user_id,
            'ins' => $invoice_payment->ins,
        ], $data_items);

        $invoice_payment->items()->whereNotIn('id', array_map(fn($v) => (int) $v['id'], $data_items))->delete();
        foreach ($data_items as $key => $value) {
            $payment_item = InvoicePaymentItem::firstOrNew(['id' => $value['id']]);
            $payment_item->fill($value);
            $payment_item->save();
        }

        // compute gain/loss on foreign exchange
        if ($frx_rate > 1) {
            $invoice_payment->update(['frx_amount' => $invoice_payment->items()->sum('frx_amount')]);
            $invoice_amount = 0;
            foreach ($invoice_payment->items as $item) {
                if ($item->invoice) {
                    $invoice_amount += ($item->invoice->frx_rate * $item['frx_rate']);
                }
            }
            $invoice_payment['frx_gain'] = $invoice_payment->frx_amount - $invoice_amount;
            $invoice_payment['invoice_amount'] = $invoice_amount;
        }
        
        // invoice payment accounting
        $invoice_payment['customer_deposit'] = true;
        $invoice_payment['pre_payment'] = !$data_items;
        $this->invoice_payment_acc($invoice_payment, 'update');

        if ($invoice_payment) {
            DB::commit();
            return $invoice_payment;
        }
    }

    /**
     * For deleting the respective model from storage
     *
     * @param InvoicePayment $payment
     * @throws GeneralException
     * @return bool
     */
    public function delete(InvoicePayment $invoice_payment)
    {
        DB::beginTransaction();
        
        $this->invoice_payment_acc($invoice_payment, 'delete');
        
        $invoice_payment->items()->delete();
        if ($invoice_payment->delete()) {
            DB::commit(); 
            return true;
        }         
    }
}
