<?php

namespace App\Repositories\Focus\purchaseorder;

use App\Models\purchaseorder\Purchaseorder;
use App\Exceptions\GeneralException;
use App\Models\account\Account;
use App\Models\assetequipment\Assetequipment;
use App\Models\items\PurchaseorderItem;
use App\Models\transaction\Transaction;
use App\Models\transactioncategory\Transactioncategory;
use App\Repositories\BaseRepository;

use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;

/**
 * Class PurchaseorderRepository.
 */
class PurchaseorderRepository extends BaseRepository
{
    /**
     * Associated Repository Model.
     */
    const MODEL = Purchaseorder::class;

    /**
     * This method is used by Table Controller
     * For getting the table data to show in
     * the grid
     * @return mixed
     */
    public function getForDataTable()
    {
        $q = $this->query()->where('document_type','requisition_stock')->orwhere('document_type','lpo_expense');

        $q->when(request('supplier_id'), function ($q) {
            $q->where('supplier_id', request('supplier_id'));
        })->when(request('status'), function ($q) {
            $q->where('status', request('status'));     
        });

        return $q->get();
    }
    public function getForExpensePo()
    {
        $q = $this->query()->where('document_type', 'lpo_expense');

        $q->when(request('supplier_id'), function ($q) {
            $q->where('supplier_id', request('supplier_id'));
        })->when(request('status'), function ($q) {
            $q->where('status', request('status'));     
        });

        return $q->get();
    }

    /**
     * For Creating the respective model in storage
     *
     * @param array $input
     * @throws GeneralException
     * @return bool
     */
    public function create(array $input)
    {
        //  dd($input);
        DB::beginTransaction();

        $order = $input['order'];
        
        foreach ($order as $key => $val) {
            $rate_keys = [
                'subtotal', 'taxable', 'total'
                //,'frx_taxable','frx_total','frx_subtotal'
            ];
            if (in_array($key, ['date', 'due_date'], 1))
                $order[$key] = date_for_database($val);
            if (in_array($key, $rate_keys, 1)) 
                $order[$key] = floatval(str_replace(',', '', $val));
        }
        $order['frx_taxable'] = $order['taxable'] * $order['currency_rate'];
        $order['frx_subtotal'] = $order['subtotal'] * $order['currency_rate'];
        // dd($order);
        $order['frx_tax'] = $order['tax'] * $order['currency_rate'];
        $order['frx_total'] = $order['total'] * $order['currency_rate'];
        $order['status'] = 'pending';
        
        
        $tid = Purchaseorder::where('ins', $order['ins'])->max('tid');
        if ($order['tid'] <= $tid) $order['tid'] = $tid+1;
        $result = Purchaseorder::create($order);

        $order_items = $input['order_items'];

        foreach ($order_items as $i => $item) {
            $item['type'] = 'stock';
            foreach ($item as $key => $val) {
                if (in_array($key, ['purchase_price', 'itemtax','tax_rate', 'item_total'], 1))
                    $item[$key] = floatval(str_replace(',', '', $val));
                if ($item['type'] == 'expense' && empty($item['uom'])) $item['uom'] = 'Lot';
            }
            $item['tax'] = $item['itemtax'];
            $item['total'] = $item['item_total'];
            //,'foreign_taxable','foreign_total','foreign_subtotal','foreign_tax','foreign_price'
            // $item['name'] = $item['description'];
            unset($item['itemtax'],$item['item_total']);
            $frx = currency_conversion($item['qty'], $item['purchase_price'],$item['tax_rate'],$result->currency_rate);
            // dd($order_items);
            // dd($frx['frx_tax']);
            if($result->lpo_include){
                $req_id = $item['requisition_item_id'];
            }else{
                $req_id = 0;
            }
            // // append modified order_items
            $order_items[$i] = array_replace($item, [
                'ins' => $result->ins,
                'user_id' => $result->user_id,
                'invoice_id' => $result->id,
                'requisition_item_id' => $req_id,
                'frx_total' => $frx['frx_total'],
                'frx_unit_cost' => $frx['frx_unit_price'],
                'frx_tax' => $frx['frx_tax'],
                'frx_subtotal' => $frx['frx_subtotal'],
                'frx_taxable' => $frx['frx_taxable'],
            ]);
            //dd($order_items);

        }
        // dd($order_items);
        PurchaseorderItem::insert($order_items);

        
        
        if ($result) {
            DB::commit();
            return $result;   
        }

        DB::rollBack();
        throw new GeneralException(trans('exceptions.backend.purchaseorders.create_error'));
    }

    /**
     * For updating the respective Model in storage
     *
     * @param Purchaseorder $purchaseorder
     * @param  $input
     * @throws GeneralException
     * return bool
     */
    public function update($purchaseorder, array $input)
    {
        //  dd($input);
        DB::beginTransaction();

        $order = $input['order'];
        foreach ($order as $key => $val) {
            $rate_keys = [
                'subtotal', 'taxable', 'total'
            ];    
            if (in_array($key, ['date', 'due_date'], 1)) 
                $order[$key] = date_for_database($val);
            if (in_array($key, $rate_keys, 1)) 
                $order[$key] = floatval(str_replace(',', '', $val));
        }
        $order['frx_taxable'] = $order['taxable'] * $order['currency_rate'];
        $order['frx_subtotal'] = $order['subtotal'] * $order['currency_rate'];
        $order['frx_tax'] = $order['tax'] * $order['currency_rate'];
        $order['frx_total'] = $order['total'] * $order['currency_rate'];

        // dd($order);
        $purchaseorder->update($order);

        $order_items = $input['order_items'];
        // dd($order_items);
        // delete omitted items
        $item_ids = array_map(function ($v) { return $v['id']; }, $order_items);
        $purchaseorder->products()->whereNotIn('id', $item_ids)->delete();
        // update or create new items
        foreach ($order_items as $item) {
            foreach ($item as $key => $val) {
                if (in_array($key, ['purchase_price', 'itemtax','tax_rate', 'item_total'], 1))
                    $item[$key] = floatval(str_replace(',', '', $val));
                if ($item['type'] == 'expense' && empty($input['uom'])) $input['uom'] = 'Lot';
            }
            $item['tax'] = $item['itemtax'];
            $item['total'] = $item['item_total'];
            $frx = currency_conversion($item['qty'], $item['purchase_price'],$item['tax_rate'],$purchaseorder->currency_rate);
            unset($item['itemtax'],$item['item_total']);
            $item = array_replace($item, [
                'ins' => $order['ins'],
                'user_id' => $order['user_id'],
                'invoice_id' => $purchaseorder->id,
                'frx_total' => $frx['frx_total'],
                'frx_unit_cost' => $frx['frx_unit_price'],
                'frx_tax' => $frx['frx_tax'],
                'frx_subtotal' => $frx['frx_subtotal'],
                'frx_taxable' => $frx['frx_taxable'],
            ]);
            // dd($item);
            $order_item = PurchaseorderItem::firstOrNew(['id' => $item['id']]);
            $order_item->fill($item);
            if (!$order_item->id) unset($order_item->id);
            $order_item->save();                
        }

        if ($purchaseorder) {
            DB::commit();
            return true;
        }

        DB::rollBack();
        throw new GeneralException(trans('exceptions.backend.purchaseorders.update_error'));
    }

    /**
     * For deleting the respective model from storage
     *
     * @param Purchaseorder $purchaseorder
     * @throws GeneralException
     * @return bool
     */
    public function delete($purchaseorder)
    {
        if ($purchaseorder->grn_items->count()) 
            throw ValidationException::withMessages(['Purchase order is attached to a Goods Receive Note!']);

        DB::beginTransaction();
        try {
            $purchaseorder->transactions()->delete();
            // aggregate_account_transactions();
            
            if ($purchaseorder->delete() && $purchaseorder->items->each->delete()) {
                DB::commit();
                return true;
            }
        } catch (\Throwable $th) {dd($th);
            DB::rollBack();
            throw new GeneralException(trans('exceptions.backend.purchaseorders.delete_error'));
        }     
    }


    /**
     * Post Account Transaction
     */
    protected function post_transaction($bill) 
    {
        // credit Accounts Payable (Creditors) 
        $account = Account::where('system', 'payable')->first(['id']);
        $tr_category = Transactioncategory::where('code', 'bill')->first(['id', 'code']);
        $tid = Transaction::where('ins', auth()->user()->ins)->max('tid') + 1;
        $cr_data = [
            'tid' => $tid,
            'account_id' => $account->id,
            'trans_category_id' => $tr_category->id,
            'credit' => $bill->grandttl,
            'tr_date' => $bill->date,
            'due_date' => $bill->due_date,
            'user_id' => $bill->user_id,
            'note' => $bill->note,
            'ins' => $bill->ins,
            'tr_type' => $tr_category->code,
            'tr_ref' => $bill->id,
            'user_type' => 'supplier',
            'is_primary' => 1
        ];
        Transaction::create($cr_data);

        $dr_data = array();
        // debit Inventory/Stock Account
        unset($cr_data['credit'], $cr_data['is_primary']);
        $is_stock = $bill->items()->where('type', 'Stock')->count();
        if ($is_stock) {
            $account = Account::where('system', 'stock')->first(['id']);
            $dr_data[] = array_replace($cr_data, [
                'account_id' => $account->id,
                'debit' => $bill->stock_subttl,
            ]);    
        }
        // debit Expense and Asset Account
        foreach ($bill->items as $item) {
            $subttl = $item->amount - $item->taxrate;
            if ($item->type == 'Expense') {
                $dr_data[] = array_replace($cr_data, [
                    'account_id' => $item->item_id,
                    'debit' => $subttl,
                ]);
            } elseif ($item->type == 'Asset') {
                $asset = Assetequipment::find($item->item_id);
                $dr_data[] = array_replace($cr_data, [
                    'account_id' => $asset->account_id,
                    'debit' => $subttl,
                ]);
            }
        }
        // debit Tax
        if ($bill->grandtax > 0) {
            $account = Account::where('system', 'tax')->first(['id']);
            $dr_data[] = array_replace($cr_data, [
                'account_id' => $account->id, 
                'debit' => $bill->grandtax,
            ]);
        }
        Transaction::insert($dr_data); 
        aggregate_account_transactions();
    }
}